//*****************************************************************************
// Arquivo pic_ws2812b.c - Desenvolvido por Márcio José Soares
// Módulo de controle para LEDs com WS2812
//
// Formato dos pulsos:
//
//  para 0
//  ------
//  |    |       |
//  |    |_______|
//   T0H    T0L
//
//  para 1
//  ---------
//  |       |     |
//  |       |_____|
//    T1H      T1L
//
// T0H = 0,4us, T0L = 0.85us
// T1H = 0,8us, T1L = 0,45us
// RES = >= 50us
//
// Cada LED requer 24 pulsos ou 3 bytes para o controle individual de cada LED
// interno respeitando a seguinte sequencia:
//      - 8 pulsos para Green (verde) + 8 pulsos para Red (vermelho) +
//        8 pulsos para Blue (azul))
//*****************************************************************************

//*****************************************************************************
// O Copyright deste programa está reservado para MARCIO JOSE SOARES e 
// seu conteúdo está protegido pela lei de Direitos Autorais LEI Nº 9.610,
// de 19 de Fevereiro de 1998. É estritamente proíbida a reprodução total ou
// parcial dos conteúdos aqui apresentados sem a prévia autorização por escrito
// do detentor dos seus direitos.
//*****************************************************************************

//******************************************************************************
// Arquivos incluídos no módulo
//******************************************************************************
#include <xc.h>
#include <stdio.h>
#include <stdlib.h>

#include "pic_ws2812b.h"
#include "my_defines.h"

//******************************************************************************
// Variáveis externas do módulo
//******************************************************************************
extern uint8_t myRed[QTLEDs];
extern uint8_t myGreen[QTLEDs];
extern uint8_t myBlue[QTLEDs];

//******************************************************************************
// Funções do módulo
//******************************************************************************

//******************************************************************************
// sndbit_WS2812 - envia um "bit" para o WS2812
//
// Entradas - bit (0 ou 1)
// Saídas   - nenhuma
//******************************************************************************
void sndbit_WS2812(_Bool WSbit){
    
    if(WSbit == 1){     // aproximadamente 800ns em Hi
        WSpinH;         // 125ns
        nop();          // + 125ns x 4 = 500ns
        nop();
        nop();
        //nop();
        WSpinL;         // + 125ns = 750ns
    }
    else{               // aproximadamente 400ns em Hi
        WSpinH;         // 125ns
        nop();          // + 125ns x 2 = 250ns
        //nop();          
        WSpinL;         // + 125ns = 500ns
    }
    
}

//******************************************************************************
// sndcolors_WS2812 - envia cores para WS2812
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void sndcolors_WS2812(void){

    uint8_t i;
    int8_t mybits;
    
    for(i=0;i<QTLEDs-1;i++){
        
        for(mybits = 7; mybits >=0; mybits--){
            (myGreen[i] & (1 << mybits)) ? sndbit_WS2812(1) : sndbit_WS2812(0);
        }
        
        for(mybits = 7; mybits >=0; mybits--){
            (myRed[i] & (1 << mybits)) ? sndbit_WS2812(1) : sndbit_WS2812(0);
        }
        
        for(mybits = 7; mybits >=0; mybits--){
            (myBlue[i] & (1 << mybits)) ? sndbit_WS2812(1) : sndbit_WS2812(0);
        }
        
    }
}

//******************************************************************************
// walk_WS2812 - efeito caminhar para WS2812
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void walk_WS2812(void){
    
    uint8_t i;
    
    for(i=0;i<QTLEDs-1;i++){
        
        myRed[i]   = myRed[i+1];
        myGreen[i] = myGreen[i+1];
        myBlue[i]  = myBlue[i+1];
       
    }
    
    myRed[QTLEDs-1]    = myRed[0];
    myGreen[QTLEDs-1]  = myGreen[0];
    myBlue[QTLEDs-1]   = myBlue[0];
    
}

//******************************************************************************
// initW_WS2812 - configura as cores nas variáveis para modo walk
//
// Entradas - cor desejada
// Saídas   - nenhuma
//******************************************************************************
void initW_WS2812(uint8_t cor){
    
    uint8_t i;
    
    for(i=0;i<QTLEDs;i++){              //
        if(cor <= BLUE){
            if(cor == RED){
                if(i <= LIM0){             // 
                    myGreen[i] = 0;     // os primeiros são vermelhos
                    myRed[i] = 0x40;    // 
                    myBlue[i] = 0;
                }
                else{
                    myGreen[i] = 0;     // limpa o restante
                    myRed[i] = 0;       // 
                    myBlue[i] = 0;
                }
            }
            else if(cor == GREEN){
                if(i <= LIM0){             // 
                    myGreen[i] = 0x40;  // os primeiros são verdes
                    myRed[i] = 0;       // 
                    myBlue[i] = 0;
                }
                else{
                    myGreen[i] = 0;     // limpa o restante
                    myRed[i] = 0;       // 
                    myBlue[i] = 0;
                }
            }
            else if(cor == BLUE){
                if(i <= LIM0){           // 
                    myGreen[i] = 0;     // os primeiros são azuis
                    myRed[i] = 0;    // 
                    myBlue[i] = 0x40;
                }
                else{
                    myGreen[i] = 0;     // limpa o restante
                    myRed[i] = 0;       // 
                    myBlue[i] = 0;
                }
            }
        }
        else{
            if(i < LIM0){                   // 
                myGreen[i] = 0;             // os primeiros são vermelhos
                myRed[i] = 0x40;            // 
                myBlue[i] = 0;
            }
            else if(i >= LIM0 && i < LIM1){ // 
                myGreen[i] = 0x40;          // os próximos são verdes
                myRed[i] = 0;               // 
                myBlue[i] = 0;
            }
            else if(i >= LIM1 && i < LIM2){ // 
                myGreen[i] = 0;             // os próximos são azuis
                myRed[i] = 0;               // 
                myBlue[i] = 0x40;
            }
            else if(i >= LIM2 && i < LIM3){ // 
                myGreen[i] = 0x00;          // os últimos são roxo
                myRed[i] = 0x40;            // 
                myBlue[i] = 0x40;
            }
            else{
                myGreen[i] = 0;             // o "gost" é sempre vermelho
                myRed[i] = 0x40;            // 
                myBlue[i] = 0;
            }
        }
    }
    
}


//******************************************************************************
// clear_WS2812 - apaga todos os LEDs
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void clear_WS2812(void){
    
    uint8_t i;
    
    for(i=0; i<QTLEDs;i++){
        myRed[i] = 0;              
        myGreen[i] = 0; 
        myBlue[i] = 0;
    }
}

//******************************************************************************
// RGB_WS2812 - acende todos os LEDs com a mesma cor (RGB))
//
// Entradas - 0 - vermelho
//            1 - verde
//            2 - azul
// Saídas   - nenhuma
//******************************************************************************
void RGB_WS2812(uint8_t cor){
    
    uint8_t i,r,g,b;
    
    switch(cor){
        case RED:
            r = 0x40;
            g = 0x00;
            b = 0x00;
            break;
        case GREEN:
            r = 0x00;
            g = 0x40;
            b = 0x00;
            break;
        case BLUE:
            r = 0x00;
            g = 0x00;
            b = 0x40;
            break;
        default:
            r = 0x40;
            g = 0x40;
            b = 0x40;
            break;
    }
    
    for(i=0; i<QTLEDs;i++){
            myRed[i] = r;              
            myGreen[i] = g; 
            myBlue[i] = b;
    }
    
}

//******************************************************************************
// blink_WS2812 - pisca todos os LEDs com a mesma cor (RGB))
//
// Entradas - 0 - vermelho
//            1 - verde
//            2 - azul
// Saídas   - nenhuma
//******************************************************************************
void blink_WS2812(uint8_t cor){
    
    clear_WS2812();
    sndcolors_WS2812();
    __delay_ms(250);
    if(cor <= BLUE)
        RGB_WS2812(cor);
    else
        initW_WS2812(RAINBOW);
    sndcolors_WS2812();
    __delay_ms(250);    

}
