//******************************************************************************
// Projeto PIC bar LED PC com PIC12F1822
// 
// Permite controlar a barra de LEDs do PC (21 LEDs WS2812 5050)
//
// Desenvolvido por Eng. Marcio Jose Soares
//
// Microcontrolador - PIC12F1822
// Compilador       - XC8 Microchip 2.40
// 
// Pinos
// RA0 - pulsos para LEDs
// RA2 - chave para seleção do efeito
//
// Obs. Usa clock interno de 32MHz
//
//******************************************************************************
// Últimas alterações:
//  em 05/01/2026:
//      - inicio deste projeto
//
//  em 07/01/2026:
//      - testes com barra de LEDs com 8 LEDs foram ok!
//******************************************************************************

//*****************************************************************************
// O Copyright deste programa está reservado para MARCIO JOSE SOARES e 
// seu conteúdo está protegido pela lei de Direitos Autorais LEI Nº 9.610,
// de 19 de Fevereiro de 1998. É estritamente proíbida a reprodução total ou
// parcial dos conteúdos aqui apresentados sem a prévia autorização por escrito
// do detentor dos seus direitos.
//*****************************************************************************

//*****************************************************************************
// Bits de configuração
//*****************************************************************************
// CONFIG1
#pragma config FOSC = INTOSC    // Oscillator Selection (INTOSC oscillator: I/O function on CLKIN pin)
#pragma config WDTE = OFF       // Watchdog Timer Enable (WDT disabled)
#pragma config PWRTE = ON       // Power-up Timer Enable (PWRT enabled)
#pragma config MCLRE = OFF      // MCLR Pin Function Select (MCLR/VPP pin function is digital input)
#pragma config CP = OFF         // Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config CPD = OFF        // Data Memory Code Protection (Data memory code protection is disabled)
#pragma config BOREN = ON       // Brown-out Reset Enable (Brown-out Reset enabled)
#pragma config CLKOUTEN = ON    // Clock Out Enable (CLKOUT function is enabled on the CLKOUT pin)
#pragma config IESO = OFF       // Internal/External Switchover (Internal/External Switchover mode is disabled)
#pragma config FCMEN = ON       // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is enabled)

// CONFIG2
#pragma config WRT = OFF        // Flash Memory Self-Write Protection (Write protection off)
#pragma config PLLEN = ON       // PLL enable (4x PLL enabled)
#pragma config STVREN = ON      // Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = LO        // Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), low trip point selected.)
#pragma config DEBUG = OFF      // In-Circuit Debugger Mode (In-Circuit Debugger disabled, ICSPCLK and ICSPDAT are general purpose I/O pins)
#pragma config LVP = OFF        // Low-Voltage Programming Enable (High-voltage on MCLR/VPP must be used for programming)

//******************************************************************************
// Arquivos incluídos no módulo
//******************************************************************************
#include <xc.h>
#include <stdio.h>
#include <stdlib.h>

#include "pic_ws2812b.h"
#include "pic_eeprom.h"
#include "my_defines.h"

//******************************************************************************
// Variáveis globais do módulo
//******************************************************************************
uint8_t myRed[QTLEDs];
uint8_t myGreen[QTLEDs];
uint8_t myBlue[QTLEDs];

uint8_t efeito = 0;
uint8_t efeito_ant = 0;
volatile uint8_t autoriza = 0;

//******************************************************************************
// Funções externas do módulo
//******************************************************************************

//******************************************************************************
// Funções do módulo
//******************************************************************************

//******************************************************************************
// Função para configurar clock
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void set_clock(void){
    
    OSCCONbits.IRCF = 0b1110;           // 32MHz interno
    OSCCONbits.SPLLEN = 1;              // PLL x4 ativado
    OSCCONbits.SCS = 0b00;              // determinado em FOSC<2:0>
    
    while(!OSCSTATbits.PLLR);           // Aguarda o PLL travar/estabilizar
    
    nop(); nop(); nop(); nop(); nop();  // apenas para estabilizar
    nop(); nop(); nop(); nop(); nop();
    nop(); nop(); nop(); nop(); nop();
    nop(); nop(); nop(); nop(); nop();
    

}

//******************************************************************************
// Função para configurar o microcontrolador
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void config_pic(void){
    
    INTCON = 0x00;                  // desabilita todas as ints
    ANSELAbits.ANSA0 = 0;           // desliga analógico em RA0
    ANSELAbits.ANSA2 = 0;           // desliga analógico em RA2
    
    TRISAbits.TRISA0 = 0;           // RA0 é saída
    TRISAbits.TRISA2 = 1;           // RA2 é entrada
    
    OPTION_REGbits.nWPUEN = 0;      // habilita resistores de pull-up
    OPTION_REGbits.INTEDG = 0;      // interrupção externa na borda de descida
    WPUAbits.WPUA2 = 1;             // habilita resistor de pull-up em RA2
    INTCONbits.INTE = 1;            // habilita int externa em RA2
    INTCONbits.INTF = 0;            // limpa flag
}

//******************************************************************************
// Função trata_int_ext - trata interrupção externa
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void trata_int_ext(void){
    
    while(readKey);                 // fica enquanto pressionada
    
    efeito++;                       // incrementa efeito
    if(efeito >= EFECTS_END){       // chegou ao último
        efeito = RED_ON;            // volta ao primeiro
    }

    autoriza = 1;                   // alterou o efeito, autoriza gravar
    __delay_ms(30);                 // pequeno bounce para a tecla
    
}


//******************************************************************************
// Função principal
//
// Entradas - nenhuma
// Saídas   - nenhuma
//******************************************************************************
void main(void) {
    
    set_clock();                            // configura clock do PIC
    config_pic();                           // configura microcontrolador
        
    WSpinL;                                 // pino de pulso em LOW
    
    __delay_ms(500);                        // tempo dummy
    
    efeito = (uint8_t)read_B_eeprom(ADDREFC);   // lê efeito registrado
    if((efeito < (RED_ON-1)) || (efeito > RAINBOW_WALK))    // se não gravado ainda
        efeito = RED_ON;                    // vai para RED_ON
    
    clear_WS2812();                     // apaga LEDs nas matrizes
    sndcolors_WS2812();                 // envia cores
    __delay_ms(250);                    // tempo dummy
    
    //**************************************************************************
    // Interrupções - tratamento global
    INTCONbits.PEIE = 1;            // habilita bit da int dos periféricos
    INTCONbits.GIE = 1;             // Habilita bit da int global
    
    INTCONbits.INTF = 0;
    __delay_ms(100);
    
    while(TRUE){ 
        switch(efeito){
            case RED_ON:
                RGB_WS2812(RED);
                sndcolors_WS2812();
                break;
            case GREEN_ON:
                RGB_WS2812(GREEN);
                sndcolors_WS2812();
                break;
            case BLUE_ON:
                RGB_WS2812(BLUE);
                sndcolors_WS2812();
                break;
            case RAINBOW_ON:
                if(efeito != efeito_ant){
                    clear_WS2812();                     // apaga LEDs nas matrizes
                    initW_WS2812(RAINBOW);
                    efeito_ant = efeito;
                }
                sndcolors_WS2812();
                break;   
            case RED_BLINK:
                blink_WS2812(RED);
                break;
            case GREEN_BLINK:
                blink_WS2812(GREEN);
                break;   
            case BLUE_BLINK:
                blink_WS2812(BLUE);
                break;    
            case RAINBOW_BLINK:
                blink_WS2812(RAINBOW);
                break;
            case RED_WALK:
                if(efeito != efeito_ant){
                    initW_WS2812(RED);
                    efeito_ant = efeito;
                }
                walk_WS2812();
                sndcolors_WS2812();
                break;
            case GREEN_WALK:
                if(efeito != efeito_ant){
                    initW_WS2812(GREEN);
                    efeito_ant = efeito;
                }
                walk_WS2812();
                sndcolors_WS2812();
                break;   
            case BLUE_WALK:
                if(efeito != efeito_ant){
                    initW_WS2812(BLUE);
                    efeito_ant = efeito;
                }
                walk_WS2812();
                sndcolors_WS2812();
                break;        
            case RAINBOW_WALK:
                if(efeito != efeito_ant){
                    initW_WS2812(RAINBOW);
                    efeito_ant = efeito;
                }
                walk_WS2812();
                sndcolors_WS2812();
                break;   
            default:
                RGB_WS2812(RED);
                sndcolors_WS2812();
        }
        __delay_ms(100);
        if(autoriza == 1){
            autoriza = 0;
            write_B_eeprom(efeito,ADDREFC);  //grava
        }
        
    }
    return;
}
